{TFTimer}
{Description: Fast MMSYSTEM timer for Delphi16}
{Author:      Richard Shotbolt}
{EMAIL:       100327,2305@compuserve.com}

{Thanks are due to Immo Wache, whose TRtcTimer provided
several ideas for this component}

{Do NOT change the line below, it is essential that this unit is fixed in
memory!}

{$C FIXED, PRELOAD, PERMANENT}

unit Ftcode;

interface

uses
  WinTypes, WinProcs, Messages, Classes, Controls, Forms, MMSystem;

type
  TFTimer = class(TComponent)
  private
    FEnabled: Boolean;
    FInterval: Word;
    FNumber: Byte; {Unique number identifying a FTimer}
    FOldResolution: Word;
    FResolution: Word;
    FTimeCaps: TTimeCaps;
    procedure UpdateTimer;
    procedure Loaded; override;
    procedure WndProc(var Message: TMessage);
    function GetCountdown: Integer;
    procedure SetCountdown(Value: Integer);
    procedure SetEnabled(Value: Boolean);
    procedure SetInterval(Value: Word);
    function GetMaxTime: Word;
    function GetMinTime: Word;
    function GetOnTimer: TNotifyEvent;
    procedure SetOnTimer(Value: TNotifyEvent);
    procedure SetResolution(Value: Word);
    procedure SetDummyWord(Value: Word);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Countdown: Integer read GetCountdown write SetCountdown default -1;
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property Interval: Word read FInterval write SetInterval default 1000;
    property MaxTime: Word read GetMaxTime write SetDummyWord;
    property MinTime: Word read GetMinTime write SetDummyWord;
    property OnTimer: TNotifyEvent read GetOnTimer write SetOnTimer;
    property Resolution: Word read FResolution write SetResolution default 10;
  end;

procedure Register;

implementation

{The variables and arrays below live in the unit's data segment and are
fixed in memory}

var
  i: Byte;
  FNoOfTimers: Byte;
  FCountdowns: Array[1..10] of Integer;
  FTimerAssigned: Array[1..10] of Boolean;
  FOnTimers: Array[1..10] of TNotifyEvent;
  FTimerIDs: Array[1..10] of Word;
  FWindowHandle: HWND;

{ ************************************************************************* }

procedure TimeCallBack(uTimerID, uMessage: Word; dwUser, dw1, dw2: LongInt); far;
begin
{This is the MMTIMER callback. It is usually placed in a DLL, but the
technique used here (courtesy of Immo Wache) seems to be reliable now that
the timer message is posted to the queue}
asm
  {set DSeg to DSeg of this unit}
  push DS;
  mov DS, Word(dwUser);
  end;
{Post a timer message to the dummy window.
This is one of the few things you ARE
allowed to do here}
PostMessage(FWindowHandle, WM_TIMER, uTimerID, 0);
{Get the DSeg back}
asm
  pop DS;
  end;
end;

{ ************************************************************************* }

function MaxOf(Value1, Value2: Word): Word;
{Return greater of two values}
begin
if Value1 >= Value2 then
  Result := Value1
else
  Result := Value2;
end;

{ ************************************************************************* }

constructor TFTimer.Create(AOwner: TComponent);
begin
if FNoOfTimers >= 10 then
  {Size of arrays limit no. of Ftimers to 10}
  raise EOutOfResources.Create('Maximum of 10 FTimers!');
inherited Create(AOwner);
{Find first unassigned FTimer and assign it}
Inc(FNoOfTimers);
for i := 1 to 10 do
  begin
  if not FTimerAssigned[i] then
    begin
    FTimerAssigned[i] := True;
    FTimerIDs[i] := 0;
    FOnTimers[i] := nil;
    FNumber := i;
    Break;
    end;
  end;
FInterval := 1000;
FResolution := 5;
FEnabled := True;
{Make a dummy window to fire PostMessages at}
FWindowHandle := AllocateHWND(WndProc);
end;

{ ************************************************************************* }

procedure TFTimer.Loaded;
{Update the Ftimer after creation}
begin
inherited Loaded;
UpdateTimer;
end;

{ ************************************************************************* }

procedure TFTimer.WndProc(var Message: TMessage);
{This is similar to the TTimer WndProc handler. In FTimer it gets its
WM_TIMER message from the callback via the message queue}
begin
with Message do
  if Msg = WM_TIMER then
    {Protect OnTimer against crashes}
    try
      begin
      for i := 1 to 10 do
        begin
        {Find which FTimer posted the message}
        if (FTimerAssigned[i]) and (Message.WParam = FTimerIDs[i])
          and (Assigned(FOnTimers[i])) then
          begin
          if FCountdowns[i] > 0 then
            begin
            {Countdown mode}
            Dec(FCountdowns[i]);
            {Trick to use the correct event handler}
            FOnTimers[i](Self);
            end
          else if FCountdowns[i] < 0 then
            {Continuous mode}
            FOnTimers[i](Self);
          {If countdown zero, timer message is trashed}
          Break;
          end;
        end;
      end;
    except
      Application.HandleException(Self);
    end
  else
    {Unlikely, but just in case ...}
    Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
end;

{ ************************************************************************* }

procedure TFTimer.UpdateTimer;
{Update all the MMTIMER stuff}
begin
if FTimerIDs[FNumber] <> 0 then
  begin
  {stop old timer events}
  timeKillEvent(FTimerIDs[FNumber]);
  timeEndPeriod(FOldResolution);
  FTimerIDs[FNumber] := 0;
  end;
{check for new timer event}
if (FInterval <> 0) and (FResolution <> 0)
  and FEnabled and Assigned(FOnTimers[FNumber]) then
  begin
  {start new timer events}
  timeBeginPeriod(FResolution);
  FOldResolution := FResolution;
  FTimerIDs[FNumber] := timeSetEvent(FInterval, FResolution, TimeCallBack,
    LongInt(DSeg), TIME_PERIODIC);
{           ^^^^ send DSeg of this unit to callback-function
            Thanks again to Immo Wache}
  if FTimerIDs[FNumber] = 0 then
    begin
    {Create timer event failed}
    timeEndPeriod(FOldResolution);
    raise EOutOfResources.Create('Timer event was not created');
    end;
  end;
end;

{ ************************************************************************* }

function TFTimer.GetMaxTime: Word;
begin
{Returns max time in ms. Usually 65535.}
if timeGetDevCaps(@FTimeCaps, SizeOf(TTimeCaps)) = 0 then
  Result := FTimeCaps.wPeriodMax;
end;

{ ************************************************************************* }

function TFTimer.GetMinTime: Word;
begin
{Returns min time in ms. Does not guarantee system will not
run out of resources!}
if timeGetDevCaps(@FTimeCaps, SizeOf(TTimeCaps)) = 0 then
  Result := FTimeCaps.wPeriodMin;
end;

{ ************************************************************************* }

procedure TFTimer.SetEnabled(Value: Boolean);
{Enable/disable Ftimer}
begin
FEnabled := Value;
UpdateTimer;
end;
 
{ ************************************************************************* }

function TFTimer.GetCountdown: Integer;
begin
Result := FCountdowns[FNumber];
end;

{ ************************************************************************* }

procedure TFTimer.SetCountdown(Value: Integer);
begin
FCountdowns[FNumber] := Value;
UpdateTimer;
end;

{ ************************************************************************* }

procedure TFTimer.SetInterval(Value: Word);
begin
FInterval := MaxOf(Value, GetMinTime);
UpdateTimer;
end;

{ ************************************************************************* }

procedure TFTimer.SetResolution(Value: Word);
begin
FResolution := MaxOf(Value, GetMinTime);
UpdateTimer;
end;

{ ************************************************************************* }

function TFTimer.GetOnTimer: TNotifyEvent;
begin
Result := FOnTimers[FNumber];
end;

{ ************************************************************************* }

procedure TFTimer.SetOnTimer(Value: TNotifyEvent);
begin
FOnTimers[FNumber] := Value;
UpdateTimer;
end;

{ ************************************************************************* }

procedure TFTimer.SetDummyWord(Value: Word);
begin
{Dummy write necessary so property can be read}
end;

{ ************************************************************************* }

destructor TFTimer.Destroy;
begin
If FNoOfTimers <> 0 then
  begin
  Dec(FNoOfTimers);
  {Disable and deallocate the current FTimer}
  FEnabled := False;
  UpdateTimer;
  FTimerAssigned[FNumber] := False;
  end;
{Any FTimers left ?}
if (FNoOfTimers = 0) and (FWindowHandle <> 0) then
  {Trash the dummy window}
  DeAllocateHWND(FWindowHandle);
inherited Destroy;
end;

{ ************************************************************************* }

procedure Register;
begin
RegisterComponents('System', [TFTimer]);
end;

{ ************************************************************************* }

begin
{First load initialises global variables and arrays}
FNoOfTimers := 0;
FWindowHandle := 0;
for i := 1 to 10 do
  begin
  FTimerAssigned[i] := False;
  FCountdowns[i] := -1;
  FOnTimers[i] := nil;
  FTimerIDs[i] := 0;
  end;

end.
